/*
 * Crestron Electronics
 * 
 * KeyButton uiHal interface
 *
 * Raymond Kuschan
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/input/matrix_keypad.h>
#include <linux/delay.h>
#include <linux/signal.h>
#include <asm/siginfo.h>        //siginfo
#include <linux/platform_device.h>
#include <linux/fsl_devices.h>


#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#include <linux/crestron.h>

#define PROC_CPUTEMP_FILENAME           "cpuTemperature"
#define PROCFS_MAX_SIZE     1024

#define TEMPERATURE_EVENT               183  //overloading input.h (#define KEY_F13  183)
#define SCANRATE 5  //Rate to check temperature

/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *CpuTemp_File;

/** * The buffer (2k) for this module */
char gCpuTemp_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
unsigned long gCpuTemp_buffer_size = 0;

extern struct platform_device ahci_fsl_device;

struct cputemp *st_cputemp;
extern int sata_init(struct device *dev);
extern ssize_t sata_ahci_current_tmp(struct device *dev, struct device_attribute *devattr, char *buf);

//extern struct mutex *cpuTempMutex; //mutex lock for gCpufreq stat
	
extern int uiHal_setCpuTemp(int buf);

/** * This funtion is called when the /proc hardkey file is read */
static ssize_t CpuTemp_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  static int finished = 0;
  char * p = gCpuTemp_buffer;
  int value;


  /* needed to stop from continuously printing */
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

  mutex_lock(&st_cputemp->cpuTempMutex);

   p += sprintf ( p, "CPU_SATA_TEMP=%d\n" , st_cputemp->CpuTemp);
   p += sprintf ( p, "PMIC_TEMP=%d\n" , st_cputemp->PmicTemp);

  mutex_unlock(&st_cputemp->cpuTempMutex);

   gCpuTemp_buffer_size = p-gCpuTemp_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gCpuTemp_buffer, gCpuTemp_buffer_size) ) {
                return -EFAULT;
        }

  return gCpuTemp_buffer_size;
}

static ssize_t CpuTemp_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{

        printk(KERN_ERR "Error: Write not supported for Temperature\n");
	
        return 0;
}
/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}
/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int CpuTemp_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int CpuTemp_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_CpuTemp_File = {
        .read    = CpuTemp_read,
        .write   = CpuTemp_write,
        .open    = CpuTemp_open_procfs,
        .release = CpuTemp_close_procfs,
};
/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
        .permission = module_permission,        /* check for permissions */
};

static void updateTempStatus(void)
{

	char buf[255];
	int tmpVal;
	struct device_attribute *devattr;
	int tempChgd=0;

//	printk("RJK: calling updateTempstatus\n");

	//read temp of CPU
	sata_ahci_current_tmp( &ahci_fsl_device.dev, devattr, buf);

	sscanf(buf, "%d", &tmpVal);

  	mutex_lock(&st_cputemp->cpuTempMutex);

	if ( tmpVal != st_cputemp->CpuTemp )
	{
		st_cputemp->CpuTemp=tmpVal;
		//printk(KERN_ERR "temp changed to: %s\n", buf);
		tempChgd=true;
	}

	//read PMIC temp
//	if ( buf != st_cputemp->PmicTemp )
//	{
//		buf=st_cputemp->pmicTemp;
//		tempChgd=true;
//	}

  	mutex_unlock(&st_cputemp->cpuTempMutex);
		
	//send an event to uiHall layer so it can read the new temp from /proc.
	if (tempChgd)
	{
		input_report_key(st_cputemp->EvInput, TEMPERATURE_EVENT, 1);
		input_sync(st_cputemp->EvInput);
		input_report_key(st_cputemp->EvInput, TEMPERATURE_EVENT, 0);
		input_sync(st_cputemp->EvInput);
	}

	//return 0;

}

static void cpuTemp_work(struct work_struct *work)
{
        const int interval = HZ * SCANRATE; //check TEMP every SCANRATE seconds 1HZ=1 second


        //printk(KERN_ERR "%s\n", __func__);

        updateTempStatus();
        schedule_delayed_work(&st_cputemp->monitor_work, interval);
}


int cpuTemperatureInit(void)
{
	int ret=0;
	

	st_cputemp = kzalloc(sizeof(*st_cputemp), GFP_KERNEL);
        if (!st_cputemp) {
                ret = -ENOMEM;
                goto fail_info;
        }

	st_cputemp->CpuTemp=0;
	st_cputemp->PmicTemp=0;
	
    	mutex_init(&st_cputemp->cpuTempMutex);

	st_cputemp->EvInput = input_allocate_device();
	if (!st_cputemp->EvInput)
                {
                        ret = -ENOMEM;
                        printk(KERN_ERR "%s() - ERROR: Could not allocate temperature input device.\n", __FUNCTION__);
                        goto fail_info;
                }

	st_cputemp->EvInput->name="cpuTempEvent";
	set_bit(EV_KEY,    st_cputemp->EvInput->evbit);
	set_bit(TEMPERATURE_EVENT,    st_cputemp->EvInput->keybit);

	ret = input_register_device(st_cputemp->EvInput);
                if (0 != ret)
                {
                        printk(KERN_ERR "%s() - ERROR: Could not register input device.\n", __FUNCTION__);
                        input_free_device(st_cputemp->EvInput);
                        goto fail_info;
                }

	/* create the /proc file for CpuTemp bitmask */
        CpuTemp_File = create_proc_entry(PROC_CPUTEMP_FILENAME, 0644, NULL);
        if (CpuTemp_File == NULL){
                printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
                       PROC_CPUTEMP_FILENAME);
                ret = -ENOMEM;
                goto fail_info;
        }
        else
        {
          //CpuTemp_File->owner = THIS_MODULE;
          CpuTemp_File->proc_iops = &Inode_Ops_File;
          CpuTemp_File->proc_fops = &File_Ops_CpuTemp_File;
          CpuTemp_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
          CpuTemp_File->uid = 0;
          CpuTemp_File->gid = 0;
          CpuTemp_File->size = 80;
       }

	/*initialize SATA so we can read the temp */
	sata_init(&ahci_fsl_device.dev);

	/* monitoring work */
        INIT_DELAYED_WORK_DEFERRABLE(&st_cputemp->monitor_work, cpuTemp_work);
        schedule_delayed_work(&st_cputemp->monitor_work, HZ*40);

        goto success;

fail_info:
        kfree(st_cputemp);
success:
        return ret;

}

static int __init cpuTemp_init(void)
{
        int ret;

        ret = cpuTemperatureInit();
        return ret;
}
module_init(cpuTemp_init);

static void __exit cpuTemp_exit(void)
{
        //exit
}
module_exit(cpuTemp_exit);

MODULE_AUTHOR("Raymond Kuschan");
MODULE_DESCRIPTION("Crestron Driver");
MODULE_LICENSE("GPL v2");

